let sidebarVisible = false;
let activeMode = 'outline'; // 'outline' | 'build'
let treeData = []; // Array of { id, title, url, children: [] }
let selectedNodeId = null; // The node currently being built upon
let drilledNodeId = null; // For drill mode - which node we're drilling into
let scrollPosition = 0; // Persist scroll position

// --- Initialization ---

// Listen for messages from Popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'toggle_sidebar') {
        toggleSidebar().then(() => {
            sendResponse({ status: 'Sidebar toggled' });
        });
        return true; // Keep channel open for async response
    }
});


// Initialize on load
(async function init() {
    // Check if sidebar was previously open or if we have state
    const state = await chrome.storage.local.get([
        'kb_sidebarVisible', 'kb_treeData', 'kb_selectedNodeId',
        'kb_activeMode', 'kb_drilledNodeId', 'kb_scrollPosition'
    ]);

    if (state.kb_treeData) treeData = state.kb_treeData;
    if (state.kb_selectedNodeId) selectedNodeId = state.kb_selectedNodeId;
    if (state.kb_activeMode) activeMode = state.kb_activeMode;
    if (state.kb_drilledNodeId) drilledNodeId = state.kb_drilledNodeId;
    if (state.kb_scrollPosition) scrollPosition = state.kb_scrollPosition;

    if (state.kb_sidebarVisible) {
        await injectSidebar();
        const sidebar = document.getElementById('kb-crawler-sidebar');
        if (sidebar) {
            sidebar.style.display = 'flex';
            sidebarVisible = true;
            renderTree();
            updateUIState();
            // Restore scroll position
            const treeContainer = document.getElementById('kb-tree-container');
            if (treeContainer && scrollPosition) {
                treeContainer.scrollTop = scrollPosition;
            }
        }
    }
})();

// --- Sidebar Injection & UI ---

async function injectSidebar() {
    if (document.getElementById('kb-crawler-sidebar')) return;

    const sidebarHtmlUrl = chrome.runtime.getURL('sidebar.html');
    const sidebarCssUrl = chrome.runtime.getURL('sidebar.css');

    // Load CSS
    if (!document.querySelector(`link[href="${sidebarCssUrl}"]`)) {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = sidebarCssUrl;
        document.head.appendChild(link);
    }

    // Load HTML
    const response = await fetch(sidebarHtmlUrl);
    const html = await response.text();

    const container = document.createElement('div');
    container.innerHTML = html;
    document.body.appendChild(container.firstElementChild);

    setupSidebarEvents();
}

async function toggleSidebar() {
    await injectSidebar();
    const sidebar = document.getElementById('kb-crawler-sidebar');
    sidebarVisible = !sidebarVisible;
    sidebar.style.display = sidebarVisible ? 'flex' : 'none';

    // Persist visibility
    chrome.storage.local.set({ kb_sidebarVisible: sidebarVisible });

    if (sidebarVisible) {
        renderTree();
        updateUIState();
    }
}

function setupSidebarEvents() {
    document.getElementById('kb-close-btn').onclick = toggleSidebar;

    document.getElementById('kb-mode-outline').onclick = () => setMode('outline');
    document.getElementById('kb-mode-build').onclick = () => setMode('build');

    document.getElementById('kb-clear-btn').onclick = () => {
        if (confirm('Clear entire outline?')) {
            treeData = [];
            selectedNodeId = null;
            saveState();
            renderTree();
        }
    };

    document.getElementById('kb-crawl-btn').onclick = startCrawl;
    document.getElementById('kb-upload-btn').onclick = uploadToNucleusKB;
    document.getElementById('kb-export-btn').onclick = exportState;
    document.getElementById('kb-structure-btn').onclick = exportStructure;
    document.getElementById('kb-import-btn').onclick = importStructure;
}

function exportState() {
    const data = {
        timestamp: new Date().toISOString(),
        treeData: treeData,
        selectedNodeId: selectedNodeId,
        activeMode: activeMode
    };

    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `kb-crawler-backup-${new Date().toISOString().slice(0, 10)}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    showNotification('✓ Full backup saved!');
}

// Export just structure (no content) - much smaller file for tracking
function exportStructure() {
    function stripContent(nodes) {
        return nodes.map(node => ({
            id: node.id,
            title: node.title,
            url: node.url,
            uploaded: node.uploaded || false,
            children: node.children ? stripContent(node.children) : []
        }));
    }

    const data = {
        timestamp: new Date().toISOString(),
        type: 'structure-only',
        treeData: stripContent(treeData)
    };

    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `kb-structure-${new Date().toISOString().slice(0, 10)}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    showNotification('✓ Structure saved (lightweight)!');
}

// Import structure from file
function importStructure() {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = '.json';
    input.onchange = async (e) => {
        const file = e.target.files[0];
        if (!file) return;

        try {
            const text = await file.text();
            const data = JSON.parse(text);

            if (data.treeData) {
                treeData = data.treeData;
                selectedNodeId = data.selectedNodeId || null;
                activeMode = data.activeMode || 'outline';
                saveState();
                renderTree();
                updateUIState();

                // Count uploaded vs pending
                let uploaded = 0, pending = 0;
                function count(nodes) {
                    nodes.forEach(n => {
                        if (n.uploaded) uploaded++;
                        else pending++;
                        if (n.children) count(n.children);
                    });
                }
                count(treeData);

                showNotification(`✓ Imported! ${uploaded} uploaded, ${pending} pending`);
            } else {
                alert('Invalid backup file format');
            }
        } catch (err) {
            alert('Error reading file: ' + err.message);
        }
    };
    input.click();
}

function setMode(mode) {
    activeMode = mode;
    // Clear selection when switching to outline mode (allows "exiting" build mode)
    if (mode === 'outline') {
        selectedNodeId = null;
    }
    // Re-enable capture when explicitly choosing a mode
    const captureCheckbox = document.getElementById('kb-capture-enabled');
    if (captureCheckbox) {
        captureCheckbox.checked = true;
    }
    saveState();
    updateUIState();
}

function updateUIState() {
    const outlineBtn = document.getElementById('kb-mode-outline');
    const buildBtn = document.getElementById('kb-mode-build');
    const statusBar = document.getElementById('kb-status-bar');

    if (activeMode === 'outline') {
        outlineBtn.classList.add('active');
        buildBtn.classList.remove('active');
        statusBar.textContent = 'Click links to add to Root';
        document.body.style.cursor = 'crosshair';
    } else {
        outlineBtn.classList.remove('active');
        buildBtn.classList.add('active');

        if (selectedNodeId) {
            const node = findNode(treeData, selectedNodeId);
            statusBar.textContent = `Adding children to: ${node ? node.title : 'Unknown'}`;
        } else {
            statusBar.textContent = 'Select a node in the tree to start building';
        }
        document.body.style.cursor = selectedNodeId ? 'copy' : 'default';
    }
}

// --- Tree Logic ---

function findNode(nodes, id) {
    for (const node of nodes) {
        if (node.id === id) return node;
        if (node.children) {
            const found = findNode(node.children, id);
            if (found) return found;
        }
    }
    return null;
}

// Enhanced tree rendering with collapsible nodes
function renderTree() {
    const container = document.getElementById('kb-tree-container');
    if (!container) return;

    // Clear existing content
    container.innerHTML = '';

    if (treeData.length === 0) {
        container.innerHTML = `
            <div class="kb-empty-state">
                <p><strong>${activeMode === 'outline' ? 'Outline' : 'Build'} Mode</strong> is active.</p>
                <p>Click links on the page to build your Table of Contents.</p>
            </div>
        `;
        updateCrawlButton();
        return;
    }

    // Render each root node
    treeData.forEach(node => {
        container.appendChild(createTreeNode(node, 0));
    });

    updateCrawlButton();
}

function createTreeNode(node, depth) {
    const nodeDiv = document.createElement('div');
    nodeDiv.className = 'kb-tree-node';
    nodeDiv.dataset.nodeId = node.id;

    // Node content wrapper
    const contentDiv = document.createElement('div');
    contentDiv.className = 'kb-node-content';
    if (node.id === selectedNodeId) {
        contentDiv.classList.add('selected-for-build');
    }
    contentDiv.style.paddingLeft = `${depth * 16 + 8}px`;

    // Expand/collapse icon
    const expandIcon = document.createElement('span');
    expandIcon.className = 'kb-expand-icon';
    expandIcon.style.cssText = 'width: 14px; display: inline-block; cursor: pointer; user-select: none;';
    if (node.children && node.children.length > 0) {
        expandIcon.textContent = node.expanded !== false ? '▼' : '▶';
        expandIcon.onclick = (e) => {
            e.stopPropagation();
            toggleNodeExpansion(node.id);
        };
    } else {
        expandIcon.textContent = '•';
        expandIcon.style.cursor = 'default';
    }

    // Title (editable on double-click)
    const titleSpan = document.createElement('span');
    titleSpan.className = 'kb-node-title';
    titleSpan.textContent = node.title;
    titleSpan.title = node.url;
    titleSpan.ondblclick = () => startEditingTitle(node.id);

    // Actions (delete button)
    const actionsDiv = document.createElement('div');
    actionsDiv.className = 'kb-node-actions';

    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'kb-action-btn';
    deleteBtn.textContent = '×';
    deleteBtn.title = 'Delete';
    deleteBtn.onclick = (e) => {
        e.stopPropagation();
        if (confirm(`Delete "${node.title}"${node.children?.length ? ' and all children' : ''}?`)) {
            deleteNode(node.id);
        }
    };

    actionsDiv.appendChild(deleteBtn);

    // Click handler for node selection
    contentDiv.onclick = () => selectNode(node.id);

    contentDiv.appendChild(expandIcon);
    contentDiv.appendChild(titleSpan);
    contentDiv.appendChild(actionsDiv);
    nodeDiv.appendChild(contentDiv);

    // Children (if expanded)
    if (node.children && node.children.length > 0 && node.expanded !== false) {
        const childrenDiv = document.createElement('div');
        childrenDiv.className = 'kb-children';
        node.children.forEach(child => {
            childrenDiv.appendChild(createTreeNode(child, depth + 1));
        });
        nodeDiv.appendChild(childrenDiv);
    }

    return nodeDiv;
}

function toggleNodeExpansion(nodeId) {
    const node = findNode(treeData, nodeId);
    if (node) {
        node.expanded = node.expanded === false ? true : false;
        saveState();
        renderTree();
    }
}

function startEditingTitle(nodeId) {
    const node = findNode(treeData, nodeId);
    if (!node) return;

    const nodeEl = document.querySelector(`[data-node-id="${nodeId}"] .kb-node-title`);
    if (!nodeEl) return;

    const currentTitle = node.title;
    const input = document.createElement('input');
    input.type = 'text';
    input.value = currentTitle;
    input.style.cssText = 'background: #333; border: 1px solid #555; color: #fff; padding: 2px 4px; border-radius: 2px; width: 100%; font-size: 12px;';

    input.onblur = () => finishEditing(nodeId, input.value);
    input.onkeydown = (e) => {
        if (e.key === 'Enter') {
            finishEditing(nodeId, input.value);
        } else if (e.key === 'Escape') {
            renderTree();
        }
    };

    nodeEl.replaceWith(input);
    input.focus();
    input.select();
}

function finishEditing(nodeId, newTitle) {
    const node = findNode(treeData, nodeId);
    if (node && newTitle.trim()) {
        node.title = newTitle.trim();
        saveState();
    }
    renderTree();
}

function updateCrawlButton() {
    const btn = document.getElementById('kb-crawl-btn');
    if (!btn) return;

    const count = countAllNodes(treeData);
    btn.textContent = `Crawl (${count})`;
    btn.disabled = count === 0;

    // Update Upload button
    const uploadBtn = document.getElementById('kb-upload-btn');
    if (uploadBtn) {
        const hasCrawledContent = treeData.some(node => hasCrawledData(node));
        uploadBtn.disabled = !hasCrawledContent;
    }
}

function hasCrawledData(node) {
    if (node.crawled || node.markdown) return true;
    if (node.children) {
        return node.children.some(child => hasCrawledData(child));
    }
    return false;
}

function countAllNodes(nodes) {
    let count = nodes.length;
    nodes.forEach(node => {
        if (node.children) {
            count += countAllNodes(node.children);
        }
    });
    return count;
}

function addNode(url, title) {
    const newNode = {
        id: crypto.randomUUID(),
        title: title || url,
        url: url,
        children: []
    };

    if (activeMode === 'outline') {
        // Add to root
        treeData.push(newNode);
    } else if (activeMode === 'build' && selectedNodeId) {
        // Add to selected parent
        const parent = findNode(treeData, selectedNodeId);
        if (parent) {
            parent.children.push(newNode);
        } else {
            alert('Parent node not found. Switch to Outline Mode to add roots.');
            return;
        }
    } else {
        alert('Select a node first to build upon.');
        return;
    }

    saveState();
    renderTree();

    // Visual feedback
    showNotification(`Added: ${newNode.title}`);
}

function deleteNode(id) {
    function removeFrom(nodes) {
        const idx = nodes.findIndex(n => n.id === id);
        if (idx !== -1) {
            nodes.splice(idx, 1);
            return true;
        }
        for (const node of nodes) {
            if (removeFrom(node.children)) return true;
        }
        return false;
    }

    removeFrom(treeData);
    if (selectedNodeId === id) selectedNodeId = null;
    saveState();
    renderTree();
}

function selectNode(id) {
    selectedNodeId = id;

    // If in build mode, we are now building for this node
    // If in outline mode, maybe we just want to highlight it? 
    // Let's auto-switch to build mode if they click a node? 
    // User requested: "hit the section item... page loads... start populating"

    // Navigate to the node's URL
    const node = findNode(treeData, id);
    if (node && node.url && node.url !== window.location.href) {
        window.location.href = node.url;
    }

    // Auto-switch to build mode for convenience?
    // "back on the extension panel, the user has an option for build mode, when they turn this on, and hit the section item..."
    // So let's respect the toggle, but update the selection.

    saveState();
    renderTree();
    updateUIState();
}

function saveState() {
    try {
        chrome.storage.local.set({
            kb_treeData: treeData,
            kb_selectedNodeId: selectedNodeId,
            kb_activeMode: activeMode
        }, () => {
            if (chrome.runtime.lastError) {
                console.error('Save state failed:', chrome.runtime.lastError);
                if (chrome.runtime.lastError.message && chrome.runtime.lastError.message.includes('QUOTA_BYTES')) {
                    showNotification('⚠️ Storage limit reached! Please upload or clear data.');
                }
            }
        });
    } catch (e) {
        console.error('Save state exception:', e);
    }
}



// --- Interaction ---

document.addEventListener('click', (e) => {
    if (!sidebarVisible) return;

    // Don't intercept clicks inside the sidebar
    const sidebar = document.getElementById('kb-crawler-sidebar');
    if (sidebar && sidebar.contains(e.target)) return;

    // Check if capture is enabled
    const captureCheckbox = document.getElementById('kb-capture-enabled');
    if (captureCheckbox && !captureCheckbox.checked) return;

    const link = e.target.closest('a');
    if (link && link.href) {
        if (activeMode === 'outline' || (activeMode === 'build' && selectedNodeId)) {
            e.preventDefault();
            e.stopPropagation();
            addNode(link.href, link.innerText.trim() || link.href);
        }
    }
}, true);

function showNotification(msg) {
    const n = document.createElement('div');
    n.textContent = msg;
    n.style.cssText = `position: fixed; top: 20px; right: 340px; background: #22c55e; color: white; padding: 8px 16px; border-radius: 4px; z-index: 2147483647; font-family: sans-serif; font-size: 14px; box-shadow: 0 2px 5px rgba(0,0,0,0.2);`;
    document.body.appendChild(n);
    setTimeout(() => n.remove(), 2000);
}

// --- Crawling with HTML to Markdown Conversion ---

async function startCrawl() {
    const btn = document.getElementById('kb-crawl-btn');
    btn.textContent = 'Starting...';
    btn.disabled = true;

    // Initialize Turndown service
    const turndownService = new TurndownService({
        headingStyle: 'atx',
        codeBlockStyle: 'fenced',
        bulletListMarker: '-'
    });

    // Collect all nodes to crawl
    const nodesToCrawl = [];
    function collectNodes(nodes) {
        nodes.forEach(node => {
            nodesToCrawl.push(node);
            if (node.children && node.children.length > 0) {
                collectNodes(node.children);
            }
        });
    }
    collectNodes(treeData);

    let completed = 0;
    const total = nodesToCrawl.length;

    // Crawl each node
    for (const node of nodesToCrawl) {
        try {
            // Fetch the page
            const response = await fetch(node.url);
            const html = await response.text();

            // Parse HTML
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');

            // Extract main content
            const content = extractMainContent(doc);

            // Convert to markdown
            node.markdown = turndownService.turndown(content);
            node.crawled = true;

            completed++;
            btn.textContent = `Crawling... ${completed}/${total}`;

        } catch (error) {
            console.error(`Failed to crawl ${node.url}:`, error);
            node.crawled = false;
            node.error = error.message;
            completed++;
        }
    }

    saveState();

    btn.textContent = `✓ Crawled ${completed}`;
    setTimeout(() => {
        updateCrawlButton();
        btn.disabled = false;
    }, 2000);

    showNotification(`✓ Crawled ${completed} pages!`);
}

function extractMainContent(doc) {
    // Try common content selectors
    const selectors = [
        'main',
        'article',
        '[role="main"]',
        '.content',
        '.documentation',
        '.doc-content',
        '#content'
    ];

    for (const selector of selectors) {
        const el = doc.querySelector(selector);
        if (el) {
            return cleanHTML(el.cloneNode(true));
        }
    }

    // Fallback: try to find the largest content block
    const body = doc.body.cloneNode(true);
    return cleanHTML(body);
}

function cleanHTML(element) {
    // Remove unwanted elements
    const unwantedSelectors = [
        'nav', 'header', 'footer', 'aside',
        '.navigation', '.sidebar', '.menu',
        '.ad', '.advertisement', '.promo',
        'script', 'style', 'noscript',
        '.social-share', '.comments'
    ];

    unwantedSelectors.forEach(selector => {
        element.querySelectorAll(selector).forEach(el => el.remove());
    });

    return element.innerHTML;
}

// --- Upload to NucleusKB API ---

async function uploadToNucleusKB() {
    const docSetName = prompt('Enter a name for this documentation set:', 'Epic Games Documentation');
    if (!docSetName) return;

    // Get stored settings or prompt
    const state = await chrome.storage.local.get(['kb_apiUrl', 'kb_apiKey']);
    let apiUrl = state.kb_apiUrl || 'https://knowledge.place';
    let apiKey = state.kb_apiKey || '';

    apiUrl = prompt('NucleusKB API URL:', apiUrl);
    if (!apiUrl) return;

    apiKey = prompt('Enter your API Key (starts with ks_):', apiKey);
    if (!apiKey) return;

    // Save settings
    chrome.storage.local.set({ kb_apiUrl: apiUrl, kb_apiKey: apiKey });

    const btn = document.getElementById('kb-upload-btn');
    const originalText = btn.textContent;
    btn.disabled = true;

    try {
        // Chunk by TOP-LEVEL sections (each with full subtree preserved)
        // This keeps parent-child relationships intact within each chunk
        // Using small chunks to avoid Vercel serverless timeout (10s limit)
        const CHUNK_SIZE = 1;  // 1 top-level section per chunk to avoid timeouts
        const chunks = [];
        for (let i = 0; i < treeData.length; i += CHUNK_SIZE) {
            chunks.push(treeData.slice(i, i + CHUNK_SIZE));
        }

        // Count total nodes for display
        function countNodes(nodes) {
            let count = 0;
            nodes.forEach(n => {
                count++;
                if (n.children) count += countNodes(n.children);
            });
            return count;
        }
        const totalNodes = countNodes(treeData);

        console.log(`[KB Upload] Starting upload of ${totalNodes} nodes in ${chunks.length} chunks (${treeData.length} top-level sections)...`);

        let docSetId = null;
        let totalProcessed = 0;
        let allConflicts = [];
        let failedChunks = [];

        for (let i = 0; i < chunks.length; i++) {
            const chunkNodeCount = countNodes(chunks[i]);
            const sectionTitle = chunks[i][0]?.title || 'Unknown';
            btn.textContent = `Uploading ${i + 1}/${chunks.length}...`;
            console.log(`[KB Upload] Chunk ${i + 1}/${chunks.length}: "${sectionTitle}" (${chunkNodeCount} nodes)`);

            try {
                const response = await fetch(`${apiUrl}/api/documentation/chunk-ingest`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${apiKey}`
                    },
                    body: JSON.stringify({
                        rootTitle: docSetName,
                        nodes: chunks[i],  // Full tree structure with children!
                        chunkIndex: i,
                        totalChunks: chunks.length,
                        docSetId: docSetId
                    })
                });

                if (!response.ok) {
                    const errorText = await response.text();
                    throw new Error(`HTTP ${response.status}: ${errorText}`);
                }

                const result = await response.json();
                docSetId = result.docSetId;
                totalProcessed += result.processedCount;
                if (result.conflicts) {
                    allConflicts = allConflicts.concat(result.conflicts);
                }

                // Mark nodes in this chunk as uploaded
                if (result.processedCount > 0) {
                    function markAllUploaded(nodes) {
                        nodes.forEach(n => {
                            n.uploaded = true;
                            if (n.children) markAllUploaded(n.children);
                        });
                    }
                    markAllUploaded(chunks[i]);
                    saveState(); // Persist upload status
                }
                console.log(`[KB Upload] ✅ Chunk ${i + 1} success: ${result.processedCount} sections`);
            } catch (chunkError) {
                console.error(`[KB Upload] ❌ Chunk ${i + 1} FAILED: "${sectionTitle}"`, chunkError);
                failedChunks.push({ index: i + 1, title: sectionTitle, error: chunkError.message });
                // Continue to next chunk instead of aborting
            }
        }

        // Report results
        if (failedChunks.length > 0) {
            const failedList = failedChunks.map(f => `• ${f.title}: ${f.error}`).join('\n');
            console.error(`[KB Upload] ${failedChunks.length} chunks failed:\n${failedList}`);
            alert(`⚠️ Upload partially complete!\n\n✅ Uploaded: ${totalProcessed} sections\n❌ Failed: ${failedChunks.length} chunks\n\nFailed sections:\n${failedList}`);
        } else if (allConflicts.length > 0) {
            showNotification(`✓ Uploaded! ${allConflicts.length} conflicts need review.`);
            alert(`✓ Upload complete!\n\n${totalProcessed} sections uploaded.\n${allConflicts.length} conflicts detected.`);
        } else {
            showNotification(`✓ Uploaded ${totalProcessed} sections!`);
            alert(`✓ Successfully uploaded ${totalProcessed} sections!`);
        }

        btn.textContent = failedChunks.length > 0 ? '⚠️ Partial' : '✓ Uploaded';
        setTimeout(() => {
            btn.textContent = originalText;
            btn.disabled = false;
        }, 3000);

    } catch (error) {
        console.error('[KB Upload] Fatal error:', error);
        alert(`❌ Upload failed: ${error.message}`);
        btn.textContent = originalText;
        btn.disabled = false;
    }
}

